%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PZC Toolbox — streamlined MATLAB implementation (no duplicates)       %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% ======================================================================
%%  PZC_optimize  — high‑level driver
%% ======================================================================
function out = PZC_optimize(my_dataframe, filterFcn, PZC_OPTIONS, varargin)
    if nargin < 2 || isempty(filterFcn); filterFcn = @PZC_filter; end

    p = inputParser; p.KeepUnmatched = true;
    addParameter(p,'smoothings',[1/10 1/25 0 -1]);
    addParameter(p,'init_par',[]);
    addParameter(p,'verbosity',0);
    addParameter(p,'only_store_last',false);
    addParameter(p,'lower_tail',true);
    parse(p,varargin{:}); S = p.Results;

    tailMult = tern(S.lower_tail,1,-1);
    y        = tailMult * my_dataframe.y;

    dA = default_if_missing(PZC_OPTIONS,'diagonal_A',false);
    dB = default_if_missing(PZC_OPTIONS,'diagonal_B',true);
    rw = default_if_missing(PZC_OPTIONS,'rw_specification',false);
    if isfield(PZC_OPTIONS,'ALPHAS_EXTREME'), alphas = PZC_OPTIONS.ALPHAS_EXTREME; else, alphas = 0.05; end
    alphas = unique(alphas(:)');

    OPT_OUT = {}; oc = 1;

    for alpha = alphas
        if isempty(S.init_par)
            init = PZC_initialize_parameters(y, alpha, dA, dB, rw);
            theta0 = init.theta0; f0 = init.f0;
        else
            theta0 = S.init_par; f0 = PZC_initialize_parameters(y, alpha, dA, dB, rw).f0;
        end

        for smoothing = S.smoothings
            if S.verbosity>=1
                fprintf('\n=== alpha=%.4g  smoothing=%.4g ===\n',alpha,smoothing);
            end

            parsFun = @(th) get_parameters(th,dA,dB,rw);
            objFun  = @(th) obj_scalar(filterFcn, y, parsFun(th), f0, alpha, smoothing);

            useUnc = exist('fminunc','file') && smoothing>=0;
            if useUnc
                opts = optimoptions('fminunc','Algorithm','quasi-newton','Display',tern(S.verbosity>0,'iter','off'),'MaxIterations',50);
                [thetaStar,fval,exitflag,output] = fminunc(objFun,theta0,opts); 
            else
                opts = optimset('Display',tern(S.verbosity>0,'iter','off'),'MaxIter',tern(smoothing<0,2000,200));
                [thetaStar,fval,exitflag,output] = fminsearch(objFun,theta0,opts); 
            end

            OPT_OUT{oc,1} = struct('alpha',alpha,'smoothing',smoothing,'theta',thetaStar,'fval',fval,'output',output); 
            oc = oc + 1; theta0 = thetaStar;

            pars = parsFun(thetaStar);
            res  = filterFcn(y, pars.omega, pars.A_mat, pars.B_mat, f0, alpha, smoothing, false);
            my_dataframe.VaR = tailMult * res.VaR;
            my_dataframe.EL  = tailMult * res.EL;
            if ~S.only_store_last
                suffix = sprintf('_s%g_alpha%g',max(smoothing,0),alpha); if smoothing<0, suffix=['_Neld' suffix]; end
                my_dataframe.(sprintf('PZC_VaR%s',suffix)) = my_dataframe.VaR;
                my_dataframe.(sprintf('PZC_EL%s',suffix))  = my_dataframe.EL;
            end
        end
    end

    out = struct('my_data',my_dataframe,'optimizer_output',{OPT_OUT});
end












